'Boom example V1.0
'By David Powell
'http://www.loadcode.co.uk
'Released into the public domain

'Pengi graphic from:
'http://openclipart.org/detail/173431/pengi-by-graphicslearning-173431

'Cherry Bomb graphic from:
'http://openclipart.org/detail/183653/cartoon-bomb-by-purzen-183653

'Explosion graphic from:
'http://openclipart.org/detail/40477/explosion-by-cybergedeon

'This example does not use mask rotation or scaling, just mask to mask collision detection.
'As the collision detection is fast, many checks are made.
'On every update the following is checked:
'1) New bombs are collision checked to make sure that they do not overlap the penguin.
'2) New bombs are collision checked to make sure that they do not overlap any existing bombs.
'3) Potential penguin movement is collision checked against all the bombs to make sure that
'   the penguin cannot move though them.
'4) The penguin is collision checked against all of the explosions to see if it has been hit.

'Also of note is that the masks were generated from versions of the images that had their
'shadows removed. Therefore a collision will only register for the body of the object
'and not the shadow.

'#FAST_SYNC_PROJECT_DATA = True

Strict

Import mojo
Import brl.pool
Import collisionmask

Public
	'Summary: Start the BoomApp application
	Function Main:Int()
		New BoomApp()
		Return 0
	End
	
Public
	'Summary: The Boom application
	Class BoomApp Extends App Implements IOnLoadMaskComplete, IOnLoadImageComplete
		Private
			Const MaxBombs:Int = 50 'The maximum number of bombs on screen at once
			Const BombSpawnChance:Float = 0.05 '5% chance to spawn a bomb every update
			Const BombCountDown:Int = 60 * 5 'How long after spawning a bomb will explode (5 seconds)
			Const ExplosionTime:Int = 60 * 1 'How long the explosion should remain on screen (1 second)
			Const PengiSpeed:Int = 4 'How fast Pengi moves across the screen (in pixels per update)
			Const DeadWait:Int = 60 * 2 'How long to pause the game after Pengi is hit by an explosion
			
			Field itemsToLoad:Int = 6 'The number of images and masks to load asynchronously
			
			'The penguin image filename and image
			Const PengiImageFilename:String = "monkey://data/pengi_shadow.png"
			Field pengiImage:Image
			
			'The penguin collision mask filename and mask
			Const PengiMaskFilename:String = "monkey://data/pengi_noshadow.cmsk"
			Field pengiMask:MasterMask
			
			'The bomb image filename and image
			Const BombImageFilename:String = "monkey://data/cherrybomb.png"
			Field bombImage:Image
			
			'The bomb collision mask filename and mask
			Const BombMaskFilename:String = "monkey://data/cherrybomb.cmsk"
			Field bombMask:MasterMask
			
			'The explosion image filename and image
			Const ExplosionImageFilename:String = "monkey://data/boom.png"
			Field explosionImage:Image
			
			'The explosion collision mask filename and mask
			Const ExplosionMaskFilename:String = "monkey://data/boom.cmsk"
			Field explosionMask:MasterMask
			
			'The penguin X and Y position
			Field pengiX:Int
			Field pengiY:Int
		
			'The penguin is dead flag
			Field pengiDead:Bool = False
			
			'The number of updates remaining until the game resets
			Field pengiDeadWait:Int = DeadWait
			
			'The list of bombs currently on screen
			Field bombList:List<Bomb>
		
	Public
		'Summary: When the application is created; initialise and start loading data asynchronously
		Method OnCreate:Int()
			'Start loading the images and masks
				
			'As the images are loaded with the Image.XYPadding flag set, when the
			'masks were generated they had a border of 1 pixel trimmed so that the
			'size is the same as the image after padding removal.
			'If a sprite atlas was used then mask trimming would not be required.
		
			LoadMaskAsync(PengiMaskFilename, Mask.DefaultFlags, Self)
			LoadMaskAsync(BombMaskFilename, Mask.MidHandle, Self)
			LoadMaskAsync(ExplosionMaskFilename, Mask.MidHandle, Self)
		
			LoadImageAsync(PengiImageFilename, 1, Image.XYPadding, Self)
			LoadImageAsync(BombImageFilename, 1, Image.XYPadding | Image.MidHandle, Self)
			LoadImageAsync(ExplosionImageFilename, 1, Image.XYPadding | Image.MidHandle, Self)
			
			'The bomb pool is now initialised and the on-screen bomb list is created
			Bomb.InitPool(MaxBombs)
			bombList = New List<Bomb>
			
			SetUpdateRate(60)

			Return 0
		End
		
	Private
		'Summary: When a mask has finished loading asynchronously; process it
		Method OnLoadMaskComplete:Void(mask:MasterMask, path:String)
			'Assign masks to the correct variables as they are loaded
			Select path
				Case PengiMaskFilename
					pengiMask = mask
					
				Case BombMaskFilename
					bombMask = mask
					bombMask.SetHandle(bombMask.Width / 2, bombMask.Height / 2)
					
				Case ExplosionMaskFilename
					explosionMask = mask
					explosionMask.SetHandle(explosionMask.Width / 2, explosionMask.Height / 2)
					
				Default
					Error("Unknown mask loaded '" + path + "'.")
			End
			
			'Reduce the load count as each mask is loaded
			itemsToLoad -= 1
		End
		
	Private
		'Summary: When an image has finished loading asynchronously; process it
		Method OnLoadImageComplete:Void(image:Image, path:String, source:IAsyncEventSource)
			'Assign images to the correct variables as they are loaded
			Select path
				Case PengiImageFilename
					pengiImage = image
					
				Case BombImageFilename
					bombImage = image
					
				Case ExplosionImageFilename
					explosionImage = image
					
				Default
					Error("Unknown image loaded '" + path + "'.")
			End
			
			'Reduce the load count as each image is loaded
			itemsToLoad -= 1
		End
		
	Public
		'Summary: Update the current state of the application
		Method OnUpdate:Int()

			If itemsToLoad > 0
				'If there are still items to load, keep updating the asynchronous events
				UpdateAsyncEvents()
				Return 0
			ElseIf itemsToLoad = 0
				'When all items have loaded reset the game state
				itemsToLoad = -1
				ResetGame()
			EndIf
		
			'If pengi is dead then keep waiting
			If pengiDead = True
				pengiDeadWait -= 1
				If pengiDeadWait <= 0
					'At the end of the wait, reset the game state
					ResetGame()
				EndIf
				
				Return 0
			EndIf
			
			'Create a new bomb
			If bombList.Count() < MaxBombs
				If Rnd() < BombSpawnChance
					Local xPos:Int = Rnd(bombImage.Width() / 2, DeviceWidth() -bombImage.Width() / 2)
					Local yPos:Int = Rnd(bombImage.Height() / 2, DeviceHeight() -bombImage.Height() / 2)
					
					'Default the overlap flag to false
					Local overlap:Bool = False
					
					'Check to make sure that the new bomb does not overlap with the penguin
					If pengiMask.CollideMask(pengiX, pengiY, bombMask, xPos, yPos) = True
						overlap = True
					EndIf
					
					'Check to make sure that the new bomb does not overlap with any other existing bombs
					If overlap = False
						For Local checkBomb:Bomb = EachIn bombList
							If bombMask.CollideMask(xPos, yPos, bombMask, checkBomb.x, checkBomb.y) = True
								overlap = True
								Exit
							EndIf
						Next
					EndIf
					
					'If there is no overlap then the new bomb can be created
					If overlap = False
						Local bomb:Bomb = Bomb.Create(xPos, yPos, BombCountDown)
						bombList.AddLast(bomb)
					EndIf
				EndIf
			EndIf
			
			'Check for movement to the right
			Local moveRight:Bool = False
			If KeyDown(KEY_RIGHT) = True Then moveRight = True
			If MouseDown() = 1 And MouseX() > pengiX + pengiImage.Width() Then moveRight = True
			
			'If the way right is not blocked by any bombs then move the penguin
			If moveRight = True And pengiX < DeviceWidth() -pengiImage.Width()
				Local blocked:Bool = False
				For Local bomb:Bomb = EachIn bombList
					If pengiMask.CollideMask(pengiX + PengiSpeed, pengiY, bombMask, bomb.x, bomb.y) = True
						blocked = True
						Exit
					EndIf
				Next
				If blocked = False Then pengiX += PengiSpeed
			EndIf
			
			'Check for movement to the left
			Local moveLeft:Bool = False
			If KeyDown(KEY_LEFT) = True Then moveLeft = True
			If MouseDown() = 1 And MouseX() < pengiX Then moveLeft = True
			
			'If the way left is not blocked by any bombs then move the penguin
			If moveLeft = True And pengiX > 0
				Local blocked:Bool = False
				For Local bomb:Bomb = EachIn bombList
					If pengiMask.CollideMask(pengiX - PengiSpeed, pengiY, bombMask, bomb.x, bomb.y) = True
						blocked = True
						Exit
					EndIf
				Next
				If blocked = False Then pengiX -= PengiSpeed
			Endif
			
			'Check for movement downwards
			Local moveDown:Bool = False
			If KeyDown(KEY_DOWN) = True Then moveDown = True
			If MouseDown() = 1 And MouseY() > pengiY + pengiImage.Height() Then moveDown = True
			
			'If the way down is not blocked by any bombs then move the penguin	
			If moveDown = True And pengiY < DeviceHeight() -pengiImage.Height()
				Local blocked:Bool = False
				For Local bomb:Bomb = EachIn bombList
					If pengiMask.CollideMask(pengiX, pengiY + PengiSpeed, bombMask, bomb.x, bomb.y) = True
						blocked = True
						Exit
					EndIf
				Next
				If blocked = False Then pengiY += PengiSpeed
			EndIf
			
			'Check for movement upwards
			Local moveUp:Bool = False
			If KeyDown(KEY_UP) = True Then moveUp = True
			If MouseDown() = 1 And MouseY() < pengiY Then moveUp = True
			
			'If the way up is not blocked by any bombs then move the penguin	
			If moveUp = True And pengiY > 0
				Local blocked:Bool = False
				For Local bomb:Bomb = EachIn bombList
					If pengiMask.CollideMask(pengiX, pengiY - PengiSpeed, bombMask, bomb.x, bomb.y) = True
						blocked = True
						Exit
					EndIf
				Next
				If blocked = False Then pengiY -= PengiSpeed
			EndIf
			
			'Check the status of each bomb
			pengiDead = False
			For Local bomb:Bomb = EachIn bombList
				'Reduce the countdown
				bomb.countdown -= 1
				
				'If the countdown is zero or below, then we have an explosion
				If bomb.countdown <= 0
					'Check to see if the penguin has collided with the explosion
					If pengiMask.CollideMask(pengiX, pengiY, explosionMask, bomb.x, bomb.y) = True
						pengiDead = True
					EndIf
				
					'If the end of the explosion has been reached then remove the bomb
					If bomb.countdown < - ExplosionTime
						bombList.Remove(bomb)
						bomb.Destroy()
					EndIf
				EndIf
			Next
			
			Return 0
		End
		
	Private
		'Summary: Reset the state of the game, ready for a new game
		Method ResetGame:Void()
			pengiDead = False 'Reset the penguin dead flag
			pengiDeadWait = DeadWait 'Set the dead wait time back to the maximum
			pengiX = DeviceWidth() / 2 - pengiImage.Width() / 2 'Centre the penguin on the screen horizontally
			pengiY = DeviceHeight() / 2 - pengiImage.Height() / 2 'Centre the penguin on the screen vertically
			
			'Return all bomb objects back to the bomb pool
			For Local bomb:Bomb = EachIn bombList
				bomb.Destroy()
			Next
			
			'Clear the list of on-screen bombs
			bombList.Clear()
			
			'Set the random number seed to the current time to randomise the next game
			Seed = Millisecs()
		End
		
	Public
		'Summary: Render the current state of the application
		Method OnRender:Int()
			'Choose the background colour depending on the state of the penguin
			If pengiDead = False
				Cls(0, 128, 0)
			Else
				Cls(128, 0, 0)
			EndIf
			
			'If there are still asynchronous items to load, then just display the loading screen
			If itemsToLoad > - 1
				DrawText("Loading...", DeviceWidth() / 2, DeviceHeight() / 2, 0.5, 0.5)
				Return 0
			EndIf
		
			'Draw each bomb on the screen
			For Local bomb:Bomb = EachIn bombList
				If bomb.countdown > 0
					'If the countdown is greater than zero; draw a bomb
					DrawImage(bombImage, bomb.x, bomb.y)
					'bombMask.DebugDraw(bomb.x, bomb.y, 0.5)
				Else
					'If the coundown is zero or negative; draw an explosion
					DrawImage(explosionImage, bomb.x, bomb.y)
					'explosionMask.DebugDraw(bomb.x, bomb.y, 0.5)
				EndIf
			Next
			
			'Draw the penguin on the screen
			DrawImage(pengiImage, pengiX, pengiY)
			'pengiMask.DebugDraw(pengiX, pengiY, 0.5)
			
			Return 0
		End
		
	End
	
Public
	'Summary: The Bomb object
	Class Bomb
		Public
			'The attributes of a Bomb
			Field x:Int
			Field y:Int
			Field countdown:Int
			
		Private
			'Global information about all Bombs
			'A bomb pool is used so that New is not called in the main game loop
			Global bombPool:Pool<Bomb>
			
		Public
			'Summary: Initialise the bomb pool with Bomb objects
			Function InitPool:Void(maxBombs:Int)
				bombPool = New Pool<Bomb>(maxBombs)
			End
			
		Public
			'Sumary: Create a Bomb
			Function Create:Bomb(x:Int, y:Int, countdown:Int)
				'Bomb objects are retrieved from the bomb pool
		        Return bombPool.Allocate().Init(x, y, countdown)
		    End
			
		Public
			'Summary: Destroy the Bomb
			Method Destroy:Void()
				'Bomb objects are returned to the bomb pool
		        bombPool.Free(Self)
		    End
			
		Public
			'Summary: Initialse a Bomb
			Method New()
				'Nothing for now
			End
			
		Private
			'Summary: Initialise the Bomb
			Method Init:Bomb(x:Int, y:Int, countdown:Int)
				Self.x = x
				Self.y = y
				Self.countdown = countdown
				Return Self
			End
	End
	